<?php
/* --------------------------------------------------------------
 Plugin.php 2020-03-11
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Application\Plugins;

/**
 * Interface Plugin
 * @package Gambio\Core\Application\Plugins
 *
 * @Note    The interface is not final yet and is very likely to change in the future.
 */
interface Plugin
{
    /**
     * List of service providers.
     *
     * The list must be a one dimensional array, containing the full qualified
     * class names of service providers to be registered.
     *
     * A service providers responsibility is to define the relationship between objects,
     * so the system can properly setup application and domain services. They will be
     * instantiated on demand and reused on consecutive requests.
     *
     * @return array|null
     */
    public function serviceProviders(): ?array;
    
    
    /**
     * List of event listeners.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = Fqn event class name
     * value = numeric, one dimensional array with fqn event listener names
     *
     * Example:
     *
     * ```php
     * use Vendor\Library\FooEvent;
     * use Vendor\Library\FooEventListenerOne;
     * use Vendor\Library\FooEventListenerTwo;
     *
     * $eventListeners = [
     *      FooEvent::class =>
     *          [
     *              FooEventListenerOne::class,
     *              FooEventListenerTwo::class,
     *          ]
     * ]
     * ```
     *
     * @return array|null
     */
    public function eventListeners(): ?array;
    
    
    /**
     * List of command handlers.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = Fqn event class name
     * value = numeric, one dimensional array with fqn command handler names
     * Example:
     *
     * ```php
     * use Vendor\Library\FooCommand;
     * use Vendor\Library\FooCommandHandlerOne;
     * use Vendor\Library\FooCommandHandlerTwo;
     *
     * $commandHandlers = [
     *      FooCommand::class =>
     *          [
     *              FooCommandHandlerOne::class,
     *              FooCommandHandlerTwo::class,
     *          ]
     * ]
     * ```
     *
     * @return array|null
     */
    public function commandHandlers(): ?array;
    
    
    /**
     * List of HTTP-GET routes.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = route, usually beginning with /.
     * value = associative, one dimensional array with dispatch callback information.
     *
     * Example:
     * ```php
     * use Vendor\Library\MyPluginController;
     *
     * $getRoutes = [
     *      '/my-plugin'              => [MyPluginController:class, 'index'],
     *      '/my-plugin/get/{someId}' => [MyPluginController:class, 'find'],
     * ];
     * ```
     *
     * @return array|null
     */
    public function getRoutes(): ?array;
    
    
    /**
     * List of HTTP-POST routes.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = route, usually beginning with /.
     * value = associative, one dimensional array with dispatch callback information.
     *
     * Example:
     * ```php
     * use Vendor\Library\MyPluginSaveAction;
     *
     * $postRoutes = [
     *      '/my-plugin/save' => [MyPluginSaveAction:class],
     * ];
     * ```
     *
     * @return array|null
     */
    public function postRoutes(): ?array;
    
    
    /**
     * List of HTTP-PUT routes.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = route, usually beginning with /.
     * value = associative, one dimensional array with dispatch callback information.
     *
     * Example:
     * ```php
     * use Vendor\Library\MyPluginPutAction;
     *
     * $putRoutes = [
     *      '/my-plugin/save' => [MyPluginPutAction:class],
     * ];
     * ```
     *
     * @return array|null
     */
    public function putRoutes(): ?array;
    
    
    /**
     * List of HTTP-DELETE routes.
     *
     * The list must be a multidimensional array in the following format:
     * (Fqn means full qualified class name).
     * key = route, usually beginning with /.
     * value = associative, one dimensional array with dispatch callback information.
     *
     * Example:
     * ```php
     * use Vendor\Library\MyPluginDeleteAction;
     *
     * $deleteRoutes = [
     *      '/my-plugin/save' => [MyPluginDeleteAction:class],
     * ];
     * ```
     *
     * @return array|null
     */
    public function deleteRoutes(): ?array;
}